import { CommonModule } from '@angular/common';
import { Component, inject, Input, SimpleChanges } from '@angular/core';
import { ControlContainer, FormControl, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { SharedModule } from '../../../../shared/shared.module';

@Component({
  selector: 'app-check-box-input',
  standalone: true,
  imports: [ReactiveFormsModule, CommonModule, SharedModule],
  viewProviders:[
    {
      provide: ControlContainer,
      useFactory : ()=>inject(ControlContainer, { skipSelf : true })
    }
  ],
  template: `<input type="checkbox" [ngClass]="classText" [id]="idField ? idField : nameField" [formControlName]="nameField" [placeholder]="placeholder" [checked]="isChecked" />
  <label [for]="idField ? idField : nameField" class="toggle-class onoffswitch2-label"></label>

  <div *ngIf="formControl.invalid && formControl.touched" class="error-invalid">
    <span class="text-danger" *ngIf="formControl.hasError('required')">{{'This field is required' | translate}}</span>
  </div>`
})
export class CheckBoxInputComponent {
  @Input() nameField!: string;
  @Input() idField!: string;
  @Input() classText!: string;
  @Input() placeholder!: string;
  @Input() isRequired!: boolean;
  @Input() isChecked!: boolean;
  parentContainer = inject(ControlContainer);

  get parentFormGroup(){
    return this.parentContainer.control as FormGroup;
  }
  get formControl(): FormControl {
    return this.parentFormGroup.get(this.nameField) as FormControl;
  }


  ngOnInit() {
    this.parentFormGroup.addControl(this.nameField, new FormControl(this.isChecked ?? '', this.isRequired ? Validators.required : null))
  }

  
  ngOnChanges(changes: SimpleChanges): void {
    // Check if the initialSelect input has changed
    if (changes['isChecked']) {
      // Update the form control value
      this.updateFormControl();
    }
  }

  private updateFormControl(): void {
    if (this.parentFormGroup) {
      // Update the value of the form control
      const control = this.parentFormGroup.get(this.nameField);
      if (control) {
        control.setValue(this.isChecked ?? '');
        control.setValidators(this.isRequired ? Validators.required : null);
        control.updateValueAndValidity();
      }
    }
  }

  ngOnDestroy() {
    this.parentFormGroup.removeControl(this.nameField);
  }
}
