<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use App\Models\User;
use Auth;
use Illuminate\Support\Facades\Schema;
use Jenssegers\Agent\Agent;
use Torann\GeoIP\Facades\GeoIP;
use App\Models\EmployeeActivity;

class RoleCreateController extends Controller
{
    public function index(Request $request)
    {
        $this->authorize('Roles & Permission Access');
        $permission = Permission::get();

        $roles = Role::with('users', 'permissions');

        $searchTerm = $request['filter'] ?? null;
        $perPage    = $request['per_page'] ?? 15;
        $page       = $request['page'] ?? 1;

        if ($searchTerm) {
            $roles->where(function ($query) use ($searchTerm) {
                $columns = Schema::getColumnListing('roles');
                foreach ($columns as $column) {
                    $query->orWhere($column, 'LIKE', "%{$searchTerm}%");
                }
                $query->orWhereHas('permissions', function ($query) use ($searchTerm) {
                    $query->where('name', 'LIKE', $searchTerm . '%');
                });
                $query->orWhereHas('users', function ($query) use ($searchTerm) {
                    $query->where('name', 'LIKE', $searchTerm . '%');
                });
            });
        }
        $items = $roles->paginate($perPage);
        $data['queryList'] = $items;
        $user = User::with('usetting')->findOrFail(Auth::user()->id);
        $role = Role::where('name', $user->getRoleNames()[0])->get();
        foreach ($role as $roles) {
            $data['permissions'] = $roles->getPermissionNames();
        }

        return $data;
    }

    public function create()
    {

        $this->authorize('Roles & Permission Create');

        $permission = Permission::get()->groupBy(function ($group) {
            return $group->permissionsgroupname;
        });
        $data['permission'] = $permission;
        $permissionLen = Permission::all()->count();
        $data['permissionLen'] = $permissionLen;

        return $data;
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->authorize('Roles & Permission Create');
        $request->validate([
            'name' => 'required|unique:roles|max:255',
        ]);

        $data   =   $request->only(['name']);
        $data['guard_name'] =   'web';
        $role = Role::create($data);
        if ($request->get('permission')) {
            foreach ($request->get('permission') as $prm) {
                $role->givePermissionTo($prm);
            }
        }

        $geolocation = GeoIP::getLocation(request()->getClientIp());
        $agent = new Agent();
        $activity = new EmployeeActivity();
        $activity->user_id = Auth::user()->id;
        $activity->activity_type = 'Created a role of ' . $request->name;
        $activity->ip_address = $geolocation->ip;
        $activity->browser = $agent->browser();
        $activity->device = $agent->device();
        $activity->save();

        return response()->json(['success' => 'The role was successfully created.']);
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $this->authorize('Roles & Permission Edit');
        $role = Role::findOrFail($id);
        $data['role'] = $role;
        $permissionLen = Permission::all()->count();
        $data['permissionLen'] = $permissionLen;
        $permissions = Permission::all()->groupBy(function ($group) {
            return $group->permissionsgroupname;
        });
        $data['permission'] = $permissions;
        $permissionNames = $role->getPermissionNames();
        $data['permissionNames'] = $permissionNames;
        return $data;
    }


    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $this->authorize('Roles & Permission Edit');

        $role = Role::find($id);

        $role = Role::find($id);
        if ($role->name == $request->name) {
            if ($request->get('permission')) {
                foreach ($request->get('permission') as $value) {
                    $permission[] = $value;
                }
            }

            $role->syncPermissions($request->get('permission'));
        } else {
            $request->validate([
                "name" => "required|max:255|unique:roles,name",
            ]);

            $role->update($request->only(['name']));
            if ($request->get('permission')) {
                foreach ($request->get('permission') as $value) {
                    $permission[] = $value;
                }
            }
            $role->syncPermissions($request->get('permission'));
        }

        $geolocation = GeoIP::getLocation(request()->getClientIp());
        $agent = new Agent();
        $activity = new EmployeeActivity();
        $activity->user_id = Auth::user()->id;
        $activity->activity_type = 'Updated a role';
        $activity->ip_address = $geolocation->ip;
        $activity->browser = $agent->browser();
        $activity->device = $agent->device();
        $activity->save();

        return response()->json(['success' => 'The role has been successfully updated.']);
    }
}
