<?php

namespace App\Http\Controllers;

use Log;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Validator;
use Stichoza\GoogleTranslate\GoogleTranslate;

// class TranslateController extends Controller
// {
//     public function translate(Request $request)
//     {
//         $validated = $request->validate([
//             'text' => 'required|string',
//             'target_lang' => 'required|string',
//         ]);

//         try {
//             $translator = new GoogleTranslate();
//             $translator->setSource();
//             $translator->setTarget($validated['target_lang']);
//             $translatedText = $translator->translate($validated['text']);
//             $translatedTitle = $translator->translate($request->title);


//             return response()->json([
//                 'success' => true,
//                 'translated_text' => $translatedText,
//                 'translated_title' => $translatedTitle,
//             ]);
//         } catch (\Exception $e) {
//             return response()->json([
//                 'success' => false,
//                 'message' => 'Translation failed.',
//             ], 500);
//         }
//     }
// }

class TranslateController extends Controller
{


    public function translate(Request $request)
    {
        $user = $request->user;
        $validator = Validator::make($request->all(), [
            'text' => 'required|string',
            'title' => 'nullable|string',
            'target_lang' => 'required|string',
        ]);

        if ($validator->fails()) {
            return Response::json(['errors' => $validator->errors()]);
        }

        // Get validated data
        $validated = $validator->validated();
        if ($user == 'customer') {
            
            $canTranslate = setting('cust_google_translate');           
        }
        elseif ($user == 'employee') {
           
            $canTranslate = setting('emp_google_translate');
        }

        if ($canTranslate == 'off') {
            return response()->json([
                'success' => false,
                'message' => 'Translation feature is disabled.',
            ], 403);
        }

        try {
            $translator = new GoogleTranslate();
            $translator->setSource(); 
            $translator->setTarget($validated['target_lang']); 
            // $translatedTitle = $validated['title'] ? $translator->translate(strip_tags($validated['title'])) : null;
            $translatedTitle = null;
            if (!empty($validated['title'])) {
                $translatedTitle = $translator->translate(strip_tags($validated['title']));
            }
            
            $translatedText = $this->translateHtmlContent($validated['text'], $translator);

            return response()->json([
                'success' => "Translated",
                'translated_text' => $translatedText ?: 'Translation unavailable',
                'translated_title' => $translatedTitle ?: 'Translation unavailable',
            ]);
        }catch (\GuzzleHttp\Exception\RequestException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Sorry, we are unable to translate at the moment. Please try again later.',
            ], 500);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while processing your request. Please try again later.',
            ], 500);
        }
    }

    /**
     * Translate HTML content while preserving HTML tags and inline styles.
     *
     * @param string $html
     * @param GoogleTranslate $translator
     * @return string
     */
    private function translateHtmlContent($html, GoogleTranslate $translator)
    {
       
        $dom = new \DOMDocument();
        libxml_use_internal_errors(true);
        $dom->loadHTML(mb_convert_encoding($html, 'HTML-ENTITIES', 'UTF-8')); 

       
        $xpath = new \DOMXPath($dom);
        $textNodes = $xpath->query('//text()'); 

        foreach ($textNodes as $node) {
            $originalText = $node->nodeValue;
            $translatedText = $translator->translate(strip_tags($originalText)); 
            $node->nodeValue = $translatedText; 
        }

        $nodesWithAttributes = $xpath->query('//*[@title or @alt]');
        foreach ($nodesWithAttributes as $node) {
            if ($node->hasAttribute('title')) {
                $node->setAttribute('title', $translator->translate($node->getAttribute('title')));
            }
            if ($node->hasAttribute('alt')) {
                $node->setAttribute('alt', $translator->translate($node->getAttribute('alt')));
            }
        }

        return $dom->saveHTML();
    }
}

